/* -*-c++-*-
 * This source code is proprietary of ADIT
 * Copyright (C) 2016 Advanced Driver Information Technology Joint Venture GmbH
 * All rights reserved
 *
 * Author: Vadiraj Kaamsha <vadiraj.kaamsha@in.bosch.com>
 * Author: Rudolf Dederer <rudolf.dederer@de.bosch.com>
*/

#include <osgBatchedText/BatchElementBase>
#include <osgBatchedText/BatcherBase>
#include <osgBatchedText/TextSimpleBase>

using namespace osgBatchedText;

bool BatchElementRenderInfo::operator < (const BatchElementRenderInfo& rhs) const
{
   return (_refNr < rhs._refNr)
      || ((_refNr == rhs._refNr)
           && ( (_onScreen > rhs._onScreen)
            || ((_onScreen == rhs._onScreen)
                 && ( (_alpha > rhs._alpha)
                  || ((_alpha == rhs._alpha)
                        && (_bsScreen.center().z() < rhs._bsScreen.center().z()))))));
}


bool BatchElementContainer::operator==(const BatchElementContainer& rhs) const
{
   unsigned int numBatchElements = _batchElements.size();
   bool areEqual = (numBatchElements == rhs._batchElements.size());
   for (unsigned int i = 0; areEqual && (i < numBatchElements); ++i)
   {
      areEqual = (_batchElements[i]->getObjectPointer() == rhs._batchElements[i]->getObjectPointer());
   }
   return areEqual;
}

bool BatchElementContainer::operator<(const BatchElementContainer& rhs) const
{
   unsigned int numBatchElements = _batchElements.size();
   if (numBatchElements < rhs._batchElements.size())
   {
      return true;
   }
   else if (numBatchElements == rhs._batchElements.size())
   {
      for (unsigned int i = 0; i < numBatchElements; ++i)
      { 
         uintptr_t p1 = _batchElements[i]->getObjectPointer();
         uintptr_t p2 = rhs._batchElements[i]->getObjectPointer();
         if (p1 < p2)
         {
            return true;
         }
         else if (p2 < p1)
         {
            return false;
         }
      }
   }
   return false;
}

bool BatchElementContainer::sortBasedOnDisplayOrder(const osg::ref_ptr<BatchElementContainer>& container1, const osg::ref_ptr<BatchElementContainer>& container2)
{
   return container1->_renderInfo._displayPriority == container2->_renderInfo._displayPriority ? ((container1->_renderInfo._displayOrder == container2->_renderInfo._displayOrder)
                                                                                                   && (container1->_renderInfo._bsScreen.radius() > container2->_renderInfo._bsScreen.radius()))
                                                                                                   || (container1->_renderInfo._displayOrder < container2->_renderInfo._displayOrder)
                                                                                                   : container1->_renderInfo._displayPriority < container2->_renderInfo._displayPriority;
}

bool BatchElementContainer::setVertexAttributes(BatchedVertexAttributes& vertexAttributes, BatchDrawParams& batchDrawParams) const
{
   static const osg::Vec4 defaultColor;
   bool success = true;
   unsigned int numBatchElements = _batchElements.size();
   unsigned int oldSize = vertexAttributes._vertices->size();
   for (unsigned int elem = _renderInfo._nextElementIndex; elem < numBatchElements; ++elem)
   {
      if (_batchElements[elem]->setVertexAttributes(vertexAttributes, _renderInfo, batchDrawParams, defaultColor, false))
      {
         oldSize = vertexAttributes._vertices->size();
      }
      else
      {
         vertexAttributes.resize(oldSize);
         _renderInfo._nextElementIndex = elem;
         success = false;
         break;
      }
   }
   return success;
}

unsigned int BatchElementContainer::getNumberOfVertices() const
{
   unsigned int numVertices = 0;
   for (std::vector< osg::ref_ptr<BatchElementBase> >::const_iterator itr = _batchElements.begin(); itr != _batchElements.end(); ++itr)
   {
      numVertices += (*itr)->getNumberOfVertices();
   }
   return numVertices;
}

void BatchElementContainer::calcTransVector(const osg::Matrix& mvMat)
{
   if (!_renderInfo._transVecValid)
   {
      for (std::vector< osg::ref_ptr<BatchElementBase> >::const_iterator itr = _batchElements.begin(); itr != _batchElements.end(); ++itr)
      {
         if ((*itr)->calcTransVector(mvMat, _renderInfo._transVec))
         {
            _renderInfo._transVecValid = true;
            break;
         }
      }
   }
}

void BatchElementTextBase::set(const TextSimpleBase* textSimpleBase, const osg::Vec4& color, const osg::Vec4& backdropColor,
                               unsigned int numChars, unsigned int beginOffset)
{
   _objectPointer = reinterpret_cast<uintptr_t>(textSimpleBase);
   _textSimpleBase = textSimpleBase;
   _color = color;
   _backdropColor = backdropColor;
   _numChars = numChars;
   _beginOffset = beginOffset;
}

bool BatchElementTextBase::calcTransVector(const osg::Matrix& mvMat, osg::Vec3& transVec) const
{
   bool success = (_textSimpleBase != NULL);
   if (success)
   {
      transVec = mvMat.preMult(_textSimpleBase->getOriginOffset() + _textSimpleBase->getPosition());
   }
   return success;
}

unsigned int BatchElementTextBase::getNumberOfVertices() const
{
   return ((_textSimpleBase != NULL) ? _numChars * (_textSimpleBase->getOutline() ? 8u : 4u) : 0);
}

bool BatchElementTextBase::checkGlyphsInTexture(BatchedVertexAttributes& vertexAttributes, BatchDrawParams& batchDrawParams) const
{
   bool success = true;
   if (NULL != _textSimpleBase)
   {
      BatcherBase* batcherBase = _textSimpleBase->getBatcher();

      bool areCharCodes = true;
      const std::vector<unsigned int>& charCodesOrGlyphIndices = _textSimpleBase->getCharCodesOrGlyphIndices(areCharCodes);
      const GlyphInfoContainerBase::LineBreakCharCodesOrGlyphIndices& lineBreakCharCodeOrGlyphIndex = batcherBase->getLineBreakCharCodesOrGlyphIndices(areCharCodes);

      std::vector<unsigned int>::const_iterator itrCharCodeOrGlyphIndex = charCodesOrGlyphIndices.begin();
      for (; success && (itrCharCodeOrGlyphIndex != charCodesOrGlyphIndices.end()); ++itrCharCodeOrGlyphIndex)
      {
         if (*itrCharCodeOrGlyphIndex != lineBreakCharCodeOrGlyphIndex._space)//Don't render space character
         {
            BatchEntry* entry = batcherBase->findOrAddGlyph(areCharCodes, *itrCharCodeOrGlyphIndex, false);
            success = (entry != NULL);
            if (success)
            {
               entry->_counter = batcherBase->getCounter();
               /* when glyph has been removed from texture or has been newly loaded add it to texture */
               if (!entry->_glyph._lower._inTexture)
               {
                  do
                  {
                     success = batcherBase->allocateSpaceForGlyph(entry->_glyph, *batchDrawParams._state);
                     if (!success)
                     {
                        //sort
                        batcherBase->sortGlyphContainer();
                        batcherBase->releaseSpace(*batchDrawParams._state);
                        batchDrawParams._countRetry++;
                     }
                  } while (!success && (batchDrawParams._countRetry < batcherBase->getMaxRetries()));
               }
               if (success)
               {
                  for (std::vector<osg::Vec2>::const_iterator itrTC = entry->_glyph._lower._texCoord.begin(); itrTC != entry->_glyph._lower._texCoord.end(); ++itrTC)
                  {
                     vertexAttributes._texCoord.push_back(osg::Vec3(*itrTC, 0.0f));
                  }
               }
            }
         }
      }
   }
   return success;
}

bool BatchElementTextBase::setVertexAttributes(BatchedVertexAttributes& vertexAttributes, const BatchElementRenderInfo& batchElementRenderInfo,
                                               BatchDrawParams& batchDrawParams, const osg::Vec4& color, bool overrideColor) const
{
   bool success = false;

   if ((NULL != _textSimpleBase) && checkGlyphsInTexture(vertexAttributes,  batchDrawParams))
   {
      success = true;
      bool hasOutline = _textSimpleBase->getOutline();
      unsigned int numVertices = 4u * _numChars;
      if (hasOutline)
      {
         vertexAttributes._texCoord.insert(vertexAttributes._texCoord.end(), vertexAttributes._texCoord.end() - numVertices, vertexAttributes._texCoord.end());

         vertexAttributes._colorArray.insert(vertexAttributes._colorArray.end(), numVertices,
                                             overrideColor ?  color : osg::Vec4(_backdropColor.r(),
                                                                                _backdropColor.g(),
                                                                                _backdropColor.b(),
                                                                                batchElementRenderInfo._alpha));
         vertexAttributes._transVecAttribOutline.insert(vertexAttributes._transVecAttribOutline.end(), numVertices,
                                                         osg::Vec4(batchElementRenderInfo._transVec, 1.0f));
      }
      vertexAttributes._colorArray.insert(vertexAttributes._colorArray.end(), numVertices,
                                          overrideColor ?  color : osg::Vec4(_color.r(),
                                                                             _color.g(),
                                                                             _color.b(),
                                                                             batchElementRenderInfo._alpha));

      vertexAttributes._transVecAttribOutline.insert(vertexAttributes._transVecAttribOutline.end(), numVertices,
                                                      osg::Vec4(batchElementRenderInfo._transVec, 0.0f));

      if (_textSimpleBase->getShaderType() >= BatchDrawableBase::CURVED_SCREEN)
      {
         for (unsigned int i = _beginOffset; i < (_beginOffset + _numChars); ++i)
         {
            vertexAttributes._relOffset.insert(vertexAttributes._relOffset.end(), 4u, _textSimpleBase->getRelCharPosition(i));
         }
         if (hasOutline)
         {
            vertexAttributes._relOffset.insert(vertexAttributes._relOffset.end(), vertexAttributes._relOffset.end() - numVertices, vertexAttributes._relOffset.end());
         }
      }

      unsigned endVertexOffset = 4u*(_beginOffset + _numChars);

      // Fill vertices
      if (vertexAttributes._vertices->getDataTypeSize() == sizeof(osg::Vec4h))
      {
         std::vector<osg::Vec4h>& vtrH = vertexAttributes.getVertexArrayHalfFloat()->asVector();
         for (unsigned int countVert = (4u*_beginOffset); countVert < endVertexOffset; ++countVert)
         {
            vtrH.push_back(_textSimpleBase->getVertexHalfFloat(countVert, batchElementRenderInfo._vertexCompW._hf, batchElementRenderInfo._cosPitchAngleFactor));
         }
         if (hasOutline)
         {
            vtrH.insert(vtrH.end(), vtrH.end() - numVertices, vtrH.end());
         }
      }
      else
      {
         std::vector<osg::Vec4f>& vtr = vertexAttributes.getVertexArrayFloat()->asVector();
         for (unsigned int countVert = (4u*_beginOffset); countVert < endVertexOffset; ++countVert)
         {
            vtr.push_back(_textSimpleBase->getVertexFloat(countVert, batchElementRenderInfo._vertexCompW._fl, batchElementRenderInfo._cosPitchAngleFactor));
         }
         if (hasOutline)
         {
            vtr.insert(vtr.end(), vtr.end() - numVertices, vtr.end());
         }
      }
   }

   return success;
}
